local LootEvents = require "lootevents"
local monsterutil = require "util.monsterutil"
local lume = require"util.lume"
-- This is a component added to mob enemies that spawns the actual loot
-- data generated by data from LootDropManager
local LootDropper = Class(function(self, inst)
	self.inst = inst

	self.loot_dropper_value = 0
	self.loot_drop_tags = {}

	self.loot_to_drop = {} -- k:ownerGUID, v:table (k:string (drop prefix), v: count)
	self.lucky_loot = {} -- same as loot_to_drop
	self.hasLootGenerated = false
	self.hasLootDropped = false
	self.konjurRewardProcessed = false
	self.has_priority = false

	if self.inst.tuning then
		local multiplayer_mods = self.inst.tuning.multiplayer_mods

		-- NOTE: Bosses and Miniboss loot values are determined by a grab bag in lootdropmanager.lua's OnLootDropperDeath().

		if multiplayer_mods then

			if self.inst.tuning.forced_loot_value ~= nil then
				self:SetLootDropperValue(self.inst.tuning.forced_loot_value)
			end

			-- All MAJOR and ELITE class mobs have loot priority by default.
			-- If we have a MAJOR or ELITE that we DO NOT want to have priority, set forced_loot_priority=false in its tuning.
			-- If we have a BASIC or MINOR or SWARM that we DO want to have priority, set forced_loot_priority=true in its tuning.

			if self.inst.tuning.forced_loot_priority ~= nil then
				self.has_priority = self.inst.tuning.forced_loot_priority
			elseif multiplayer_mods == "MAJOR" or multiplayer_mods == "ELITE" or multiplayer_mods == "MINIBOSS" then
				self.has_priority = true
			end

			-- print("Loot Debug:", self.inst, "Forced loot value:", self.inst.tuning.forced_loot_value, "GetLootDropperValue():", self:GetLootDropperValue(), "Forced loot priority:", self.inst.tuning.forced_loot_priority, "Has loot priority:", self.has_priority)
		end
	end
end)

local LootDropValBits <const> = 2 -- expect loot value to never be > 4

function LootDropper:OnNetSerialize()
	local e = self.inst.entity
	assert(self.loot_dropper_value <= 4, "more than 2 bits of value")
	e:SerializeUInt(self.loot_dropper_value, LootDropValBits)
end

function LootDropper:OnNetDeserialize()
	local e = self.inst.entity
	local v = e:DeserializeUInt(LootDropValBits)
	self:SetLootDropperValue(v)
end

function LootDropper:OnPostSpawn()
	monsterutil.BuildDropTable(self.inst)
end

function LootDropper:HasDropped()
	return self.hasLootDropped
end

function LootDropper:OnDeath()
	-- TheLog.ch.Networking:printf("LootDropper:OnDeath")
	local hasAddedLoot = false
	if not self.hasLootGenerated then
		self.hasLootGenerated = true
		for _, player in ipairs(AllPlayers) do
			if player:IsLocal() then
				-- this call has side effects into self.loot_to_drop, self.lucky_loot
				-- this call has side effects into player.components.lootdropmanager
				hasAddedLoot = player.components.lootdropmanager:OnLootDropperDeath(self.inst) or hasAddedLoot
			end
		end
	end

	-- Send the loot generated for these local players to remote clients so they can process
	-- the drop and generate their own loot.
	-- Local players will see the generated data processed on the entity's death via DropLoot()
	if hasAddedLoot then
		LootEvents.MakeEventGenerateLoot(self.inst, self.loot_to_drop, self.lucky_loot)
	end	
end

function LootDropper:OnDeathKonjur()
	if not self.konjurRewardProcessed then
		self.konjurRewardProcessed = true
		LootEvents.MakeEventRequestSpawnCurrency(self.inst)
	end
end

function LootDropper:AddLootToDrop(owner, loot)
	if self.loot_to_drop[owner] then
		self.loot_to_drop[owner] = lume.merge(self.loot_to_drop[owner], loot)
	else
		self.loot_to_drop[owner] = loot
	end
end

function LootDropper:AddLuckyLoot(owner, loot)
	if self.lucky_loot[owner] then
		self.lucky_loot[owner] = lume.merge(self.lucky_loot[owner], loot)
	else
		self.lucky_loot[owner] = loot
	end
end

function LootDropper:DropLoot()
	if not self.hasLootDropped then
		self.hasLootDropped = true
		LootEvents.HandleEventDropLoot(self.inst, self.loot_to_drop, self.lucky_loot)
	end
end

function LootDropper:AddLootDropTags(tags)
	-- add a table of tags. this will allow items that contain all of the tags in the table to drop.
	-- for example, if you add a table that has {'drops_cabbageroll', 'drops_common'} the dropper can only drop an item that has both of those tags.
	-- but if you add a table that just has {'drops_cabbageroll'} the dropper will be able to drop any item that has the 'drops_cabbageroll' tag, regardless of other tags the item might have.
	table.insert(self.loot_drop_tags, tags)
end

function LootDropper:OnRemoveFromEntity()
	self.inst:RemoveEventCallback("death", self._on_death)
end

function LootDropper:SetLootDropperValue(val)
	assert(val, self.inst.prefab)
	self.loot_dropper_value = val
end

function LootDropper:GetLootDropperValue()
	return self.loot_dropper_value
end

function LootDropper:HasPriority()
	return self.has_priority
end

return LootDropper
